import * as state from './state.js';
import * as ipc from './ipc.js';
import { t, playSound, navigateSound, confirmSound, cancelSound, backgroundMusic, anyModalOpen } from './utils.js';

let onProcessStartedCallback = () => {};

export function init(callbacks) {
    onProcessStartedCallback = callbacks.onProcessStarted;
}

export const carouselList = document.getElementById('carousel-list');
export const gameTitleElement = document.getElementById('game-title');
export const launchMenu = document.getElementById('launch-menu-container');
export const launchMenuTitle = document.getElementById('launch-menu-title');
export const launchMenuButtons = launchMenu.querySelectorAll('.menu-button');
export const progressOverlay = document.getElementById('progress-overlay');
export const progressStatus = document.getElementById('progress-status');
export const progressBar = document.getElementById('progress-bar');
export const progressPercent = document.getElementById('progress-percent');
export const shelfList = document.getElementById('shelf-list');
export const shelfContainer = document.querySelector('.shelf-container');
export const topGuiBar = document.getElementById('top-gui-bar');
export const bottomGuiBar = document.getElementById('bottom-gui-bar');
export const mainContent = document.querySelector('.main-content');
export const alphaJumpBar = document.getElementById('alpha-jump-bar');
export const saveStateMenu = document.getElementById('save-state-menu-container');
export const saveStateTooltip = document.getElementById('save-state-tooltip');
export const saveStateSlotsContainer = document.getElementById('save-state-slots-container');
export const topMenuItems = document.querySelectorAll('.top-menu-item');
export const lobbyMenu = document.getElementById('lobby-menu-container');
export const lobbyList = document.getElementById('lobby-list');
export const powerMenu = document.getElementById('power-menu-container');
export const powerMenuButtons = powerMenu.querySelectorAll('.menu-button');
export const audioMenu = document.getElementById('audio-menu-container');
export const audioMenuOptions = audioMenu.querySelectorAll('.modal-option-item');
export const bgmVolumeBar = document.getElementById('bgm-volume-bar');
export const bgmNameDisplay = document.getElementById('bgm-name-display');
export const displayMenu = document.getElementById('display-menu-container');
export const displayMenuOptions = displayMenu.querySelectorAll('.modal-option-item');
export const startFullscreenDisplay = document.getElementById('start-fullscreen-display');
export const languageDisplay = document.getElementById('language-display');
export const attractModeDisplay = document.getElementById('attract-mode-display');
export const networkMenu = document.getElementById('network-menu-container');
export const networkMenuOptions = networkMenu.querySelectorAll('.modal-option-item');
export const raLoginButton = document.getElementById('ra-login-button');
export const relayServerDisplay = document.getElementById('relay-server-display');
export const usernameModal = document.getElementById('username-modal-container');
export const usernameInput = document.getElementById('username-input');
export const usernameModalInputs = [document.getElementById('username-input'), ...document.getElementById('username-modal-buttons').querySelectorAll('.menu-button')];
export const configMenu = document.getElementById('config-menu-container');
export const configMenuOptions = configMenu.querySelectorAll('.modal-option-item');
export const saveSystemToggleDisplay = document.getElementById('save-system-toggle-display');
export const raToggleDisplay = document.getElementById('ra-toggle-display');
export const aspectRatioDisplay = document.getElementById('aspect-ratio-display');
export const videoFullscreenDisplay = document.getElementById('video-fullscreen-display');
export const scanlinesDisplay = document.getElementById('scanlines-display');
export const scanlinesStrengthDisplay = document.getElementById('scanlines-strength-display');
export const rewindFfDisplay = document.getElementById('rewind-ff-display');
export const saveStateModal = document.getElementById('save-state-modal-container');
export const saveStateModalTitle = document.getElementById('save-state-modal-title');
export const saveStateModalButtons = document.querySelectorAll('.save-state-modal-button');
export const raLoginModal = document.getElementById('ra-login-modal-container');
export const raUsernameInput = document.getElementById('ra-username-input');
export const raPasswordInput = document.getElementById('ra-password-input');
export const raLoginModalButtons = [document.getElementById('ra-login-save-button'), document.getElementById('ra-login-cancel-button')];
export const raLoginModalInputs = [raUsernameInput, raPasswordInput, ...raLoginModalButtons];
export const movingBgContainer = document.querySelector('.moving-bg-container');
export const notificationContainer = document.getElementById('notification-container');
export const notificationTitle = document.getElementById('notification-title');
export const notificationBody = document.getElementById('notification-body');
export const notificationButton = document.getElementById('notification-button');
export const attractModeOverlay = document.getElementById('attract-mode-overlay');

export async function fetchNotificationData() {
    try {
        const response = await fetch('https://raw.githubusercontent.com/ex-ds-dev/CBnotification/refs/heads/main/PCMiniClassicNotification');
        if (response.ok) {
            state.setNotificationData(await response.json());
        } else {
            console.error('Failed to fetch notification data, status:', response.status);
        }
    } catch (error) {
        console.error('Error fetching notification data:', error);
    }
}

export function showNotification() {
    if (!state.notificationData || !state.notificationData.enabled) return;
    notificationTitle.textContent = state.notificationData.title || '';
    notificationBody.textContent = state.notificationData.bodytext || '';
    if (state.notificationData.link && state.notificationData.link.url && state.notificationData.link.text) {
        notificationButton.textContent = state.notificationData.link.text;
        notificationButton.style.display = 'block';
        notificationButton.onclick = () => ipc.openExternalLink(state.notificationData.link.url);
    } else {
        notificationButton.style.display = 'none';
    }
    notificationContainer.classList.add('visible');
    const duration = (state.notificationData.duration_seconds || 10) * 1000;
    if (state.notificationTimeout) clearTimeout(state.notificationTimeout);
    state.setNotificationTimeout(setTimeout(hideNotification, duration));
}

export function hideNotification() {
    notificationContainer.classList.remove('visible');
}

export function renderUI() {
    carouselList.innerHTML = '';
    shelfList.innerHTML = '';
    const gamepad1PIconPath = `${state.UI_URLS.uiDirUrl}/gamepad.png`;
    const gamepad2PIconPath = `${state.UI_URLS.uiDirUrl}/gamepad-2p.png`;
    const carouselFragment = document.createDocumentFragment();
    const shelfFragment = document.createDocumentFragment();
    for (const game of state.games) {
        const coverSrc = game.coverUrl ? game.coverUrl : 'about:blank';
        const isTwoPlayer = state.twoPlayerGames.includes(game.baseName);
        const playerCountText = isTwoPlayer ? '2P' : '1P';
        const gamepadIconToUse = isTwoPlayer ? gamepad2PIconPath : gamepad1PIconPath;
        const iconClass = isTwoPlayer ? 'controller-icon controller-icon-2p' : 'controller-icon';
        const gameItem = document.createElement('div');
        gameItem.className = 'game-item';
        gameItem.innerHTML = `<img class="game-cover" src="${coverSrc}" alt="${game.baseName} Cover Art"><div class="game-info-bar"><div class="suspend-points"><div class="suspend-point"></div><div class="suspend-point"></div><div class="suspend-point"></div><div class="suspend-point"></div></div><div class="player-info"><span class="player-count">${playerCountText}</span><img class="${iconClass}" src="${gamepadIconToUse}" alt="Controller Icon"></div></div>`;
        carouselFragment.appendChild(gameItem);
        const shelfItem = document.createElement('div');
        shelfItem.className = 'shelf-item';
        if (game.coverUrl) {
            shelfItem.style.backgroundImage = `url("${game.coverUrl}")`;
        }
        shelfFragment.appendChild(shelfItem);
    }
    carouselList.appendChild(carouselFragment);
    shelfList.appendChild(shelfFragment);
}

export function updateMainContentPadding() { if (!topGuiBar || !bottomGuiBar || !mainContent) return; const topHeight = topGuiBar.offsetHeight; const bottomHeight = bottomGuiBar.offsetHeight; mainContent.style.paddingTop = `${topHeight}px`; mainContent.style.paddingBottom = `${bottomHeight}px`; }

export async function updateSuspendPointsUI(index) {
    if (index < 0 || index >= state.games.length) return;
    const gameBaseName = state.games[index].zipName.replace(/\.zip$/i, '');
    const gameItem = carouselList.children[index];
    if (!gameItem) return;

    const suspendPoints = gameItem.querySelectorAll('.suspend-point');
    const updateDots = (count) => { suspendPoints.forEach((point, i) => point.classList.toggle('filled', i < count)); };

    if (state.saveStateCache[gameBaseName] !== undefined && state.saveStateCache[gameBaseName] !== 'fetching') { updateDots(state.saveStateCache[gameBaseName]); return; }
    if (state.saveStateCache[gameBaseName] === 'fetching') return;

    state.saveStateCache[gameBaseName] = 'fetching';
    const existingStates = await ipc.manageSaveState({ action: 'scan', gameBaseName });
    const saveStateCount = existingStates.filter(st => st !== null).length;
    state.saveStateCache[gameBaseName] = saveStateCount;
    updateDots(saveStateCount);
}

export function updateSelection(newIndex, playSoundEffect = true) {
    if (state.games.length === 0) return;
    if (playSoundEffect && state.selectedIndex !== newIndex) { playSound(navigateSound); }
    const oldSelectedItem = carouselList.children[state.selectedIndex];
    if (oldSelectedItem) oldSelectedItem.classList.remove('selected');
    const oldSelectedShelfItem = shelfList.children[state.selectedIndex];
    if (oldSelectedShelfItem) oldSelectedShelfItem.classList.remove('selected');
    
    let resolvedIndex = newIndex;
    if (newIndex < 0) resolvedIndex = state.games.length - 1;
    else if (newIndex >= state.games.length) resolvedIndex = 0;
    state.setSelectedIndex(resolvedIndex);

    const newSelectedItem = carouselList.children[state.selectedIndex];
    const newSelectedShelfItem = shelfList.children[state.selectedIndex];
    if (!newSelectedItem || !newSelectedShelfItem) return;

    newSelectedItem.classList.add('selected');
    newSelectedShelfItem.classList.add('selected');
    gameTitleElement.textContent = state.games[state.selectedIndex].baseName;
    state.appSettings.lastGameIndex = state.selectedIndex;

    const windowCenter = window.innerWidth / 2;
    const itemCenter = newSelectedItem.offsetLeft + (newSelectedItem.offsetWidth / 2);
    const scrollAmount = windowCenter - itemCenter;
    carouselList.style.transform = `translateX(${scrollAmount}px)`;
    const shelfContainerCenter = shelfContainer.offsetWidth / 2;
    const shelfItemCenter = newSelectedShelfItem.offsetLeft + (newSelectedShelfItem.offsetWidth / 2);
    const shelfScrollAmount = shelfContainerCenter - shelfItemCenter;
    shelfList.style.transform = `translateX(${shelfScrollAmount}px)`;

    updateSuspendPointsUI(state.selectedIndex);
    for (let i = 1; i <= 3; i++) {
        updateSuspendPointsUI(state.selectedIndex + i);
        updateSuspendPointsUI(state.selectedIndex - i);
    }
}

export function renderAlphaBar() { alphaJumpBar.innerHTML = ''; state.ALPHABET.forEach(letter => { const letterEl = document.createElement('span'); letterEl.className = 'alpha-letter'; letterEl.textContent = letter; alphaJumpBar.appendChild(letterEl); }); }
export function updateAlphaSelection() { Array.from(alphaJumpBar.children).forEach((el, index) => { el.classList.toggle('selected', index === state.alphaSelectionIndex); }); }

export function openAlphaBar() {
    if (anyModalOpen() || state.isAlphaBarOpen || state.games.length === 0) return;
    const firstChar = state.games[state.selectedIndex].baseName.charAt(0).toUpperCase();
    const isNumeric = !isNaN(parseInt(firstChar));
    let newAlphaIndex = state.ALPHABET.indexOf(firstChar);
    if (isNumeric) newAlphaIndex = 0;
    else if (newAlphaIndex === -1) newAlphaIndex = 0;
    state.setAlphaSelectionIndex(newAlphaIndex);
    state.setAlphaBarOpen(true);
    playSound(confirmSound);
    alphaJumpBar.classList.add('visible');
    updateAlphaSelection();
}

export function closeAlphaBar(playSoundEffect = true) {
    if (!state.isAlphaBarOpen) return;
    state.setAlphaBarOpen(false);
    if (playSoundEffect) playSound(cancelSound);
    alphaJumpBar.classList.remove('visible');
}

export async function populateSaveStateSlots() {
    if (state.games.length === 0) return;
    const gameBaseName = state.games[state.selectedIndex].zipName.replace(/\.zip$/i, '');
    const fetchedStates = await ipc.manageSaveState({ action: 'scan', gameBaseName });
    state.setCurrentSaveStates(fetchedStates);
    const slots = saveStateSlotsContainer.children;
    for (let i = 0; i < slots.length; i++) {
        const slot = slots[i];
        const screenshotEl = slot.querySelector('.save-state-screenshot');
        const infoBarEl = slot.querySelector('.save-state-info-bar');
        screenshotEl.innerHTML = '';
        screenshotEl.style.backgroundImage = '';
        infoBarEl.textContent = '';
        const stateData = state.currentSaveStates[i];
        if (stateData) {
            const fileSlot = stateData.fileSlot;
            const slotSuffix = fileSlot > 0 ? `${fileSlot}` : '';
            const pngPath = `${state.APP_PATHS.statesDir}/${gameBaseName}.state${slotSuffix}.png`.replace(/\\/g, '/');
            const cacheBuster = `?t=${Date.now()}`;
            screenshotEl.style.backgroundImage = `url('file:///${pngPath}${cacheBuster}')`;
            if (stateData.mtime) {
                const date = new Date(stateData.mtime);
                const dateString = date.toLocaleDateString(undefined, { year: '2-digit', month: '2-digit', day: '2-digit' });
                const timeString = date.toLocaleTimeString(undefined, { hour: 'numeric', minute: '2-digit', hour12: true });
                infoBarEl.textContent = `${dateString}, ${timeString}`;
            }
        } else {
            screenshotEl.innerHTML = `<span class="no-data-text">${t('savestate_no_data')}</span>`;
        }
    }
}

export function updateSaveStateSelection(playSoundEffect = true) { if (playSoundEffect) playSound(navigateSound); const slots = saveStateSlotsContainer.children; for (let i = 0; i < slots.length; i++) { slots[i].classList.toggle('selected', i === state.saveStateSelectionIndex); } }
export async function openSaveStateMenu() { if (anyModalOpen() || state.isSaveStateMenuOpen) return; state.setSaveStateMenuOpen(true); playSound(confirmSound); await populateSaveStateSlots(); saveStateMenu.classList.add('visible'); saveStateTooltip.classList.add('visible'); state.setSaveStateSelectionIndex(0); updateSaveStateSelection(false); }
export function closeSaveStateMenu() { if (!state.isSaveStateMenuOpen) return; state.setSaveStateMenuOpen(false); playSound(cancelSound); saveStateMenu.classList.remove('visible'); saveStateTooltip.classList.remove('visible'); }
export function updateTopMenuSelection() { topMenuItems.forEach((item, index) => { item.classList.toggle('selected', index === state.topMenuSelectionIndex); }); }
export function focusTopMenu() { if (anyModalOpen() || state.isSaveStateMenuOpen || state.isTopMenuFocused) return; state.setTopMenuFocused(true); playSound(navigateSound); state.setTopMenuSelectionIndex(0); updateTopMenuSelection(); carouselList.children[state.selectedIndex]?.classList.remove('selected'); shelfList.children[state.selectedIndex]?.classList.remove('selected'); }
export function unfocusTopMenu(playSoundEffect = true) { if (!state.isTopMenuFocused) return; state.setTopMenuFocused(false); if (playSoundEffect) playSound(cancelSound); state.setTopMenuSelectionIndex(-1); updateTopMenuSelection(); state.setTopMenuSelectionIndex(0); carouselList.children[state.selectedIndex]?.classList.add('selected'); shelfList.children[state.selectedIndex]?.classList.add('selected'); }
export function openLaunchMenu() { if (anyModalOpen()) return; state.setMenuOpen(true); playSound(confirmSound); launchMenuTitle.textContent = state.games[state.selectedIndex].baseName; launchMenu.classList.remove('hidden'); state.setMenuSelectionIndex(0); updateMenuSelection(); }
export function closeLaunchMenu(playSoundEffect = true) { if (!state.isMenuOpen) return; state.setMenuOpen(false); if (playSoundEffect) playSound(cancelSound); launchMenu.classList.add('hidden'); }
export function updateMenuSelection() { launchMenuButtons.forEach((button, index) => { button.classList.toggle('selected', index === state.menuSelectionIndex); }); }
export function openSaveStateModal() { if (anyModalOpen()) return; state.setSaveStateModalOpen(true); playSound(confirmSound); saveStateModalTitle.textContent = t('savestate_modal_title', { slot: state.saveStateSelectionIndex + 1 }); saveStateModal.classList.remove('hidden'); state.setSaveStateModalSelectionIndex(0); updateSaveStateModalSelection(); }
export function closeSaveStateModal() { if (!state.isSaveStateModalOpen) return; state.setSaveStateModalOpen(false); playSound(cancelSound); saveStateModal.classList.add('hidden'); }
export function updateSaveStateModalSelection() { saveStateModalButtons.forEach((button, index) => { button.classList.toggle('selected', index === state.saveStateModalSelectionIndex); }); }
export function updateLobbySelection() {
    Array.from(lobbyList.children).forEach((item, index) => {
        const isSelected = index === state.lobbySelectionIndex;
        item.classList.toggle('selected', isSelected);
        if (isSelected) {
            item.scrollIntoView({ block: 'nearest' });
        }
    });
}
export function renderLobbyList(hostedGames) { lobbyList.innerHTML = ''; state.setLobbyGames(hostedGames); if (state.lobbyGames.length === 0) { lobbyList.innerHTML = `<div class="lobby-item">${t('lobbymenu_no_games')}</div>`; return; } const fragment = document.createDocumentFragment(); state.lobbyGames.forEach(game => { const item = document.createElement('div'); item.className = 'lobby-item'; item.innerHTML = `<span class="game-name">${game.game_name}</span><span class="host-name">${t('lobbymenu_hosted_by', { host: game.username })}</span>`; fragment.appendChild(item); }); lobbyList.appendChild(fragment); state.setLobbySelectionIndex(0); updateLobbySelection(); }
export async function openLobbyMenu() { if (anyModalOpen()) return; state.setLobbyOpen(true); playSound(confirmSound); lobbyList.innerHTML = `<div class="lobby-item">${t('lobbymenu_fetching')}</div>`; lobbyMenu.classList.remove('hidden'); try { const hostedGames = await ipc.getLobbyList(); renderLobbyList(hostedGames); } catch (error) { console.error("Failed to get lobby list:", error); lobbyList.innerHTML = `<div class="lobby-item">${t('lobbymenu_error')}</div>`; } }
export function closeLobbyMenu() { if (!state.isLobbyOpen) return; state.setLobbyOpen(false); playSound(cancelSound); lobbyMenu.classList.add('hidden'); state.setLobbyGames([]); }
export function updatePowerMenuSelection() { powerMenuButtons.forEach((button, index) => { button.classList.toggle('selected', index === state.powerMenuSelectionIndex); }); }
export function openPowerMenu(playSoundEffect = true) { state.setPowerMenuOpen(true); if (playSoundEffect) playSound(confirmSound); powerMenu.classList.remove('hidden'); state.setPowerMenuSelectionIndex(0); updatePowerMenuSelection(); }
export function closePowerMenu(playSoundEffect = true) { if (!state.isPowerMenuOpen) return; state.setPowerMenuOpen(false); if (playSoundEffect) playSound(cancelSound); powerMenu.classList.add('hidden'); }
export function openAudioMenu(playSoundEffect = true) { if (anyModalOpen()) return; state.setAudioMenuOpen(true); if (playSoundEffect) playSound(confirmSound); audioMenu.classList.remove('hidden'); state.setAudioMenuSelectionIndex(0); renderBgmVolumeBar(); updateAudioMenuSelection(); }
export function closeAudioMenu(playSoundEffect = true) { if (!state.isAudioMenuOpen) return; state.setAudioMenuOpen(false); if (playSoundEffect) playSound(cancelSound); audioMenu.classList.add('hidden'); ipc.saveAppSettings(state.appSettings); }
export function renderBgmVolumeBar() { bgmVolumeBar.innerHTML = ''; for (let i = 0; i < 10; i++) { const block = document.createElement('div'); block.className = i < state.appSettings.bgmVolumeLevel ? 'volume-block' : 'volume-block off'; bgmVolumeBar.appendChild(block); } }
export function updateAudioMenuSelection() { audioMenuOptions.forEach((item, index) => { item.classList.toggle('selected', index === state.audioMenuSelectionIndex); }); }
export function changeBgm(direction) { if (state.availableBgmTracks.length === 0) return; state.appSettings.currentBgmIndex = (state.appSettings.currentBgmIndex + direction + state.availableBgmTracks.length) % state.availableBgmTracks.length; const track = state.availableBgmTracks[state.appSettings.currentBgmIndex]; if (track.name === 'Music Off') { bgmNameDisplay.textContent = t('audiomenu_bgm_off'); state.appSettings.bgmPath = "off"; backgroundMusic.pause(); } else if (track.name === 'Default') { bgmNameDisplay.textContent = t('audiomenu_bgm_default'); const defaultTrack = state.availableBgmTracks.find(t => t.name === 'Default'); state.appSettings.bgmPath = "default"; if (defaultTrack && defaultTrack.path) { backgroundMusic.src = `file:///${defaultTrack.path.replace(/\\/g, '/')}`; backgroundMusic.play().catch(e => console.error("BGM failed to play:", e)); } } else { bgmNameDisplay.textContent = track.name; state.appSettings.bgmPath = track.path; backgroundMusic.src = `file:///${track.path.replace(/\\/g, '/')}`; backgroundMusic.play().catch(e => console.error("BGM failed to play:", e)); } playSound(navigateSound); }
export function openDisplayMenu(playSoundEffect = true) { if (anyModalOpen()) return; state.setDisplayMenuOpen(true); if (playSoundEffect) playSound(confirmSound); populateDisplayMenu(); displayMenu.classList.remove('hidden'); state.setDisplayMenuSelectionIndex(0); updateDisplayMenuSelection(); }
export function closeDisplayMenu(playSoundEffect = true) { if (!state.isDisplayMenuOpen) return; state.setDisplayMenuOpen(false); if (playSoundEffect) playSound(cancelSound); displayMenu.classList.add('hidden'); ipc.saveAppSettings(state.appSettings); }
export function populateDisplayMenu() {
    startFullscreenDisplay.textContent = state.appSettings.startFullscreen ? t('displaymenu_fullscreen') : t('displaymenu_windowed');
    const currentLang = state.availableLanguages.find(l => l.code === state.appSettings.language);
    languageDisplay.textContent = currentLang ? currentLang.name : 'English';
    attractModeDisplay.textContent = state.appSettings.attractModeEnabled === 'true' ? t('common_on') : t('common_off');
}
export function updateDisplayMenuSelection() { displayMenuOptions.forEach((item, index) => { item.classList.toggle('selected', index === state.displayMenuSelectionIndex); }); }
export async function openNetworkMenu(playSoundEffect = true) { if (anyModalOpen()) return; state.setNetworkMenuOpen(true); if (playSoundEffect) playSound(confirmSound); await populateNetworkMenu(); networkMenu.classList.remove('hidden'); state.setNetworkMenuSelectionIndex(0); updateNetworkMenuSelection(); }
export function closeNetworkMenu(playSoundEffect = true) { if (!state.isNetworkMenuOpen) return; state.setNetworkMenuOpen(false); if (playSoundEffect) playSound(cancelSound); if (state.currentRelayIndex !== state.initialRelayIndex) { const newConfig = state.relayOptions[state.currentRelayIndex]; ipc.setRelayConfig(newConfig); } networkMenu.classList.add('hidden'); ipc.saveAllSettings(state.appSettings); }
export async function populateNetworkMenu() { raToggleDisplay.textContent = state.appSettings.cheevos_enable === 'true' ? t('common_enable') : t('common_disable'); raLoginButton.classList.toggle('disabled', state.appSettings.cheevos_enable !== 'true'); const currentConfig = await ipc.getRelayConfig(); const foundIndex = state.relayOptions.findIndex(opt => opt.server === currentConfig.server && opt.enabled === currentConfig.enabled); const newIndex = (foundIndex !== -1) ? foundIndex : 0; state.setCurrentRelayIndex(newIndex); state.setInitialRelayIndex(newIndex); updateRelayServerDisplay(); }
export function updateNetworkMenuSelection() { networkMenuOptions.forEach((item, index) => { item.classList.toggle('selected', index === state.networkMenuSelectionIndex); }); }
export function updateRelayServerDisplay() { const currentOption = state.relayOptions[state.currentRelayIndex]; relayServerDisplay.textContent = t(currentOption.labelKey); }
export function openUsernameModal() { if (state.isUsernameModalOpen) return; closeNetworkMenu(false); state.setUsernameModalOpen(true); usernameModal.classList.remove('hidden'); usernameInput.value = state.appSettings.netplayUsername; state.setUsernameModalSelectionIndex(0); updateUsernameModalSelection(); setTimeout(() => usernameInput.focus(), 50); }
export function closeUsernameModal() { if (!state.isUsernameModalOpen) return; state.setUsernameModalOpen(false); usernameModal.classList.add('hidden'); }
export function updateUsernameModalSelection() { usernameModalInputs.forEach((el, idx) => { el.classList.toggle('selected', idx === state.usernameModalSelectionIndex); if (el.tagName === 'INPUT') el.classList.toggle('focus', idx === state.usernameModalSelectionIndex); }); }
export function openConfigMenu(playSoundEffect = true) { if (anyModalOpen()) return; state.setConfigMenuOpen(true); if (playSoundEffect) playSound(confirmSound); populateConfigMenu(); configMenu.classList.remove('hidden'); state.setConfigMenuSelectionIndex(0); updateConfigMenuSelection(); }
export function closeConfigMenu(playSoundEffect = true) { if (!state.isConfigMenuOpen) return; state.setConfigMenuOpen(false); if (playSoundEffect) playSound(cancelSound); configMenu.classList.add('hidden'); ipc.saveAllSettings(state.appSettings); }
export function populateConfigMenu() { saveSystemToggleDisplay.textContent = state.appSettings.savestate_auto_index === 'true' ? t('common_enable') : t('common_disable'); aspectRatioDisplay.textContent = state.aspectLevels[state.appSettings.aspect_ratio_index] || '4:3'; videoFullscreenDisplay.textContent = state.appSettings.video_fullscreen === 'true' ? t('common_yes') : t('common_no'); scanlinesDisplay.textContent = state.appSettings.input_overlay_enable === 'true' ? t('common_on') : t('common_off'); const strengthKey = `configmenu_strength_${state.strengthLevels[state.appSettings.input_overlay_opacity]?.toLowerCase() || 'normal'}`; scanlinesStrengthDisplay.textContent = t(strengthKey); rewindFfDisplay.textContent = state.appSettings.rewind_enable === 'true' ? t('common_yes') : t('common_no'); }
export function updateConfigMenuSelection() { configMenuOptions.forEach((item, index) => { item.classList.toggle('selected', index === state.configMenuSelectionIndex); }); }
export function openRaLoginModal() { if (state.isRaLoginModalOpen) return; closeNetworkMenu(false); state.setRaLoginModalOpen(true); raLoginModal.classList.remove('hidden'); raUsernameInput.value = state.appSettings.cheevos_username || ''; raPasswordInput.value = state.appSettings.cheevos_password || ''; state.setRaLoginModalSelectionIndex(2); updateRaLoginModalSelection(); setTimeout(() => raUsernameInput.focus(), 50); }
export function closeRaLoginModal(shouldRevertRaToggle = false) { if (!state.isRaLoginModalOpen) return; state.setRaLoginModalOpen(false); raLoginModal.classList.add('hidden'); if (shouldRevertRaToggle) { state.appSettings.cheevos_enable = "false"; } }
export function updateRaLoginModalSelection() { raLoginModalInputs.forEach((el, idx) => { el.classList.toggle('selected', idx === state.raLoginModalSelectionIndex); if (el.tagName === 'INPUT') { el.classList.toggle('focus', idx === state.raLoginModalSelectionIndex); } }); }

export function showProgressOverlay(initialMessage) {
    state.setProcessActive(true);
    onProcessStartedCallback();
    closeLaunchMenu(false);
    closeSaveStateMenu();
    closeSaveStateModal();
    movingBgContainer.classList.add('animation-paused');
    progressOverlay.classList.remove('hidden');
    progressStatus.textContent = initialMessage;
    progressBar.style.width = '0%';
    progressPercent.textContent = '';
    progressBar.classList.remove('error');
}
export function hideProgressOverlay() {
    movingBgContainer.classList.remove('animation-paused');
    progressOverlay.classList.add('hidden');
}

export function setCssAndImageSources(urls) {
    const root = document.documentElement;
    root.style.setProperty('--bg-url', `url('${urls.uiDirUrl}/bg.png')`);
    root.style.setProperty('--bg2-url', `url('${urls.uiDirUrl}/bg2.png')`);
    root.style.setProperty('--sstate-bg-url', `url('${urls.uiDirUrl}/sstate.png')`);

    document.querySelector('#top-gui-bar img').src = `${urls.uiDirUrl}/top_gui.png`;
    document.querySelector('#bottom-gui-bar img').src = `${urls.uiDirUrl}/bottom_gui.png`;
    document.querySelector('.top-menu-item[alt="In-Game Configuration"]').src = `${urls.uiDirUrl}/menu1.png`;
    document.querySelector('.top-menu-item[alt="Audio Options"]').src = `${urls.uiDirUrl}/menu2.png`;
    document.querySelector('.top-menu-item[alt="General Configuration"]').src = `${urls.uiDirUrl}/menu3.png`;
    document.querySelector('.top-menu-item[alt="Network Features"]').src = `${urls.uiDirUrl}/menu4.png`;
    document.querySelector('.top-menu-item[alt="Power Options"]').src = `${urls.uiDirUrl}/menu5.png`;
    document.querySelector('.info-bar-item img[alt="Menu"]').src = `${urls.uiDirUrl}/gamepad_up.png`;
    document.querySelector('.info-bar-item img[alt="Suspend"]').src = `${urls.uiDirUrl}/gamepad_down.png`;
    document.querySelector('.info-bar-item img[alt="Lobby"]').src = `${urls.uiDirUrl}/select.png`;
    document.querySelector('.info-bar-item img[alt="Sort"]').src = `${urls.uiDirUrl}/start.png`;
    document.querySelector('.progress-mario').src = `${urls.uiDirUrl}/mario.gif`;
    document.querySelector('#loading-overlay .loading-mario').src = `${urls.uiDirUrl}/mario.gif`;
}